<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/functions.php';

// Check if user has permission to manage suppliers
if (!hasPermission('manage_suppliers')) {
    header('Location: dashboard.php');
    exit;
}

$activePage = 'suppliers';

$success = '';
$error = '';

// Function to check if serial numbers are unique
function checkSerialNumbersUnique($serial_numbers) {
    global $pdo;
    $serials = explode(',', $serial_numbers);
    $serials = array_map('trim', $serials);
    
    // Check for duplicates in the input
    if (count($serials) !== count(array_unique($serials))) {
        return false;
    }
    
    // Check against existing serial numbers in the database
    $placeholders = str_repeat('?,', count($serials) - 1) . '?';
    $stmt = $pdo->prepare("
        SELECT COUNT(*) 
        FROM supplier_receipts 
        WHERE serial_numbers IS NOT NULL 
        AND JSON_OVERLAPS(serial_numbers, JSON_ARRAY($placeholders))
    ");
    $stmt->execute($serials);
    return $stmt->fetchColumn() === 0;
}

// Add this function at the top of the file after the existing functions
function getLastSerialNumber($prefix = '') {
    global $pdo;
    $stmt = $pdo->prepare("
        SELECT serial_numbers 
        FROM supplier_receipts 
        WHERE serial_numbers IS NOT NULL
    ");
    $stmt->execute();
    $max_number = 0;
    
    while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
        if (!empty($row['serial_numbers'])) {
            $serials = json_decode($row['serial_numbers'], true);
            if (is_array($serials)) {
                foreach ($serials as $serial) {
                    // Remove prefix if it exists
                    $number = str_replace($prefix, '', $serial);
                    // Extract numeric part
                    if (preg_match('/\d+/', $number, $matches)) {
                        $num = intval($matches[0]);
                        $max_number = max($max_number, $num);
                    }
                }
            }
        }
    }
    return $max_number;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['receive_item'])) {
    try {
        $pdo->beginTransaction();
        
        $supplier_id = intval($_POST['supplier_id']);
        $order_id = !empty($_POST['order_id']) ? intval($_POST['order_id']) : null;
        $product_ids = $_POST['product_id'];
        $quantities = $_POST['quantity'];
        $unit_prices = $_POST['unit_price'];
        $batch_numbers = $_POST['batch_number'];
        $serial_numbers_arr = $_POST['serial_numbers'] ?? [];
        $order_item_ids = $_POST['order_item_id'] ?? [];
        
        // Validate supplier
        $stmt = $pdo->prepare("SELECT id FROM suppliers WHERE id = ?");
        $stmt->execute([$supplier_id]);
        if (!$stmt->fetch()) {
            throw new Exception('Invalid supplier selected.');
        }
        
        // If this is from an order, validate the order
        if ($order_id) {
            $stmt = $pdo->prepare("SELECT id, status FROM supplier_orders WHERE id = ? AND supplier_id = ?");
            $stmt->execute([$order_id, $supplier_id]);
            $order = $stmt->fetch();
            if (!$order) {
                throw new Exception('Invalid order selected.');
            }
            if ($order['status'] !== 'approved') {
                throw new Exception('Only approved orders can be received.');
            }
        }
        
        // Process each item
        for ($i = 0; $i < count($product_ids); $i++) {
            $product_id = intval($product_ids[$i]);
            $quantity = intval($quantities[$i]);
            $unit_price = floatval($unit_prices[$i]);
            $batch_number = trim($batch_numbers[$i]);
            $serial_numbers = !empty($serial_numbers_arr[$i]) ? array_map('trim', explode(',', $serial_numbers_arr[$i])) : [];
            $order_item_id = !empty($order_item_ids[$i]) ? intval($order_item_ids[$i]) : null;
            
            // Get product category and check if serials are required
            $stmt = $pdo->prepare("
                SELECT c.require_serials, p.name 
                FROM products p 
                JOIN categories c ON p.category_id = c.id 
                WHERE p.id = ?
            ");
            $stmt->execute([$product_id]);
            $product_info = $stmt->fetch();
            if (!$product_info) {
                throw new Exception("Product not found.");
            }
            
            $require_serials = (bool)$product_info['require_serials'];
            $product_name = $product_info['name'];
            
            // Validate serial numbers if required
            if ($require_serials) {
                if (empty($serial_numbers)) {
                    throw new Exception("Serial numbers are required for product: {$product_name}.");
                }
                if (count($serial_numbers) !== $quantity) {
                    throw new Exception("Number of serial numbers must match the quantity for product: {$product_name}.");
                }
                if (count(array_unique($serial_numbers)) !== count($serial_numbers)) {
                    throw new Exception("Duplicate serial numbers are not allowed for product: {$product_name}.");
                }
                
                // Check if serial numbers already exist
                $stmt = $pdo->prepare("
                    SELECT COUNT(*) 
                    FROM product_serial_numbers 
                    WHERE serial_number IN (" . str_repeat('?,', count($serial_numbers) - 1) . "?)
                ");
                $stmt->execute($serial_numbers);
                if ($stmt->fetchColumn() > 0) {
                    throw new Exception("One or more serial numbers already exist in the system for product: {$product_name}.");
                }
            }
            
            // Insert into supplier_receipts
            // Set both status and payment_status based on receipt type
            // All receipts are recorded as pending payment - actual payment is done via Pay button in suppliers.php
            $payment_status = 'pending'; // All receipts start as unpaid credit
            $status = $order_id ? 'yes' : 'pending'; // 'yes' means stock is added, 'pending' means waiting for approval
            
            $stmt = $pdo->prepare("
                INSERT INTO supplier_receipts (
                    supplier_id, product_id, quantity, unit_price, batch_number, 
                    created_by, created_at, serial_numbers, order_id, order_item_id, payment_status, status
                ) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $supplier_id,
                $product_id,
                $quantity,
                $unit_price,
                $batch_number,
                $_SESSION['user_id'],
                !empty($serial_numbers) ? json_encode($serial_numbers) : null,
                $order_id,
                $order_item_id,
                $payment_status,
                $status
            ]);
            $receipt_id = $pdo->lastInsertId();
            
            // Add serial numbers to product_serial_numbers if required (only for order receipts, manual entries wait for approval)
            if ($require_serials && !empty($serial_numbers) && $order_id) {
                $stmt = $pdo->prepare("
                    INSERT INTO product_serial_numbers 
                    (product_id, serial_number, source_type, source_id) 
                    VALUES (?, ?, 'supplier_receipt', ?)
                ");
                foreach ($serial_numbers as $serial) {
                    $stmt->execute([$product_id, $serial, $receipt_id]);
                }
            }
            
            // Update product stock ONLY for receipts from orders (manual entries require approval via action button)
            if ($order_id) {
                $stmt = $pdo->prepare("UPDATE products SET quantity = quantity + ? WHERE id = ?");
                $stmt->execute([$quantity, $product_id]);
            }
            
            // Update order item status if from order
            if ($order_item_id) {
                $stmt = $pdo->prepare("
                    UPDATE supplier_order_items 
                    SET received_quantity = received_quantity + ?, 
                        status = CASE 
                            WHEN received_quantity + ? >= quantity THEN 'received' 
                            ELSE 'partial' 
                        END 
                    WHERE id = ?
                ");
                $stmt->execute([$quantity, $quantity, $order_item_id]);
            }
        }
        
        // If this is from an order, check if all items are received and update order status
        if ($order_id) {
            $stmt = $pdo->prepare("
                SELECT COUNT(*) as total_items, 
                       SUM(CASE WHEN status = 'received' THEN 1 ELSE 0 END) as received_items
                FROM supplier_order_items 
                WHERE order_id = ?
            ");
            $stmt->execute([$order_id]);
            $order_stats = $stmt->fetch();
            
            if ($order_stats['total_items'] == $order_stats['received_items']) {
                // All items received, mark order as completed
                $stmt = $pdo->prepare("UPDATE supplier_orders SET status = 'completed', updated_at = NOW() WHERE id = ?");
                $stmt->execute([$order_id]);
                
                // Log the action
                logSystemAction(
                    'supplier_order_completed',
                    "Order #{$order_id} completed - all items received"
                );
            }
        }
        
        $pdo->commit();
        if ($order_id) {
            $success = "Receipt created successfully! Items received from order and stock updated automatically.";
        } else {
            $success = "Receipt created successfully! Items are pending - use the 'Add to Stock' button in the Actions column to add them to inventory.";
        }
        
        header("Location: receivesupplieritem.php?success=" . urlencode($success));
        exit;
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Error creating receipt: " . $e->getMessage();
    }
}

// Handle receipt approval/rejection
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_status'])) {
    $receipt_id = intval($_POST['receipt_id']);
    $new_status = $_POST['status'];
    
    try {
        $pdo->beginTransaction();
        
        // Get the receipt data from the database instead of using the hidden input
        $stmt = $pdo->prepare("SELECT * FROM supplier_receipts WHERE id = ?");
        $stmt->execute([$receipt_id]);
        $receipt = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$receipt) {
            throw new Exception("Receipt not found");
        }

        // Update receipt status
        $stmt = $pdo->prepare("UPDATE supplier_receipts SET status = ? WHERE id = ?");
        $stmt->execute([$new_status, $receipt_id]);

        // If status is 'yes', update product stock
        if ($new_status === 'yes') {
            // Check if product exists
            $stmt = $pdo->prepare("SELECT id FROM products WHERE id = ?");
            $stmt->execute([$receipt['product_id']]);
            if (!$stmt->fetch()) {
                throw new Exception("Product not found for receipt");
            }
            $stmt = $pdo->prepare("
                UPDATE products 
                SET quantity = quantity + ?
                WHERE id = ?
            ");
            $stmt->execute([$receipt['quantity'], $receipt['product_id']]);
        }

        $pdo->commit();
        $success = "Receipt status updated successfully!";
        // Redirect to prevent form resubmission
        header('Location: receivesupplieritem.php?success=1');
        exit;
    } catch (Exception $e) {
        $pdo->rollBack();
        $error = "Error updating receipt: " . $e->getMessage() . ' (Line: ' . $e->getLine() . ')';
        error_log($error);
    }
}

// Get success message from URL parameter
if (isset($_GET['success'])) {
    if ($_GET['success'] === '1') {
    $success = "Operation completed successfully!";
    } else {
        $success = urldecode($_GET['success']);
    }
}

// Get all suppliers with their items and payment details
$stmt = $pdo->query("
    SELECT 
        s.*,
        COUNT(sr.id) as total_items,
        SUM(CASE WHEN sr.payment_status = 'paid' THEN 1 ELSE 0 END) as paid_items,
        SUM(sr.quantity * sr.unit_price) as total_amount,
        SUM(CASE WHEN sr.payment_status = 'paid' THEN sr.quantity * sr.unit_price ELSE 0 END) as paid_amount
    FROM suppliers s
    LEFT JOIN supplier_receipts sr ON s.id = sr.supplier_id
    GROUP BY s.id
    ORDER BY s.name
");
$suppliers = $stmt->fetchAll();

// Get all products with their category information
$stmt = $pdo->query("
    SELECT p.*, c.require_serials 
    FROM products p 
    LEFT JOIN categories c ON p.category_id = c.id 
    WHERE p.status = 'active' 
    ORDER BY p.name
");
$products = $stmt->fetchAll();

// Get all approved supplier orders with their items
$stmt = $pdo->query("
    SELECT 
        so.id as order_id,
        so.order_date,
        so.expected_delivery_date,
        so.notes as order_notes,
        s.id as supplier_id,
        s.name as supplier_name
    FROM supplier_orders so
    JOIN suppliers s ON so.supplier_id = s.id
    WHERE so.status = 'approved'
    ORDER BY so.order_date DESC, s.name
");
$approved_orders = $stmt->fetchAll();

// Get order items for all approved orders
$order_items = [];
if (!empty($approved_orders)) {
    $order_ids = array_column($approved_orders, 'order_id');
    $placeholders = str_repeat('?,', count($order_ids) - 1) . '?';
    
    $stmt = $pdo->prepare("
        SELECT 
            soi.id as order_item_id,
            soi.order_id,
            soi.product_id,
            soi.quantity,
            soi.received_quantity,
            soi.status as item_status,
            soi.notes as item_notes,
            p.name as product_name,
            c.require_serials
        FROM supplier_order_items soi
        JOIN products p ON soi.product_id = p.id
        LEFT JOIN categories c ON p.category_id = c.id
        WHERE soi.order_id IN ($placeholders)
        AND soi.status IN ('pending', 'partial')
        ORDER BY soi.order_id, p.name
    ");
    $stmt->execute($order_ids);
    $order_items_flat = $stmt->fetchAll();
    
    // Group items by order_id
    foreach ($order_items_flat as $item) {
        $order_items[$item['order_id']][] = $item;
    }
}

// Get the next batch number to suggest in the form
$next_batch_number_suggestion = getNextBatchNumber($pdo);

// Get the next manual batch number for manual entries
$next_manual_batch_number = getNextManualBatchNumber($pdo);

// --- New, Efficient Data Fetching and Grouping Logic ---

// Get all receipts and group by supplier
$stmt = $pdo->query("
    SELECT 
        sr.id,
        sr.supplier_id,
        s.name as supplier_name,
        sr.product_id,
        p.name as product_name,
        sr.quantity,
        sr.unit_price,
        sr.batch_number,
        sr.payment_status,
        sr.order_id,
        sr.order_item_id,
        sr.serial_numbers,
        sr.created_at,
        sr.status,
        u.username as created_by_name
    FROM supplier_receipts sr
    JOIN suppliers s ON sr.supplier_id = s.id
    JOIN products p ON sr.product_id = p.id
    JOIN users u ON sr.created_by = u.id
    ORDER BY s.name, sr.created_at DESC
");
$all_receipts = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Group receipts by supplier
$receipts_by_supplier = [];
foreach ($all_receipts as $receipt) {
    $supplier_id = $receipt['supplier_id'];
    if (!isset($receipts_by_supplier[$supplier_id])) {
        $receipts_by_supplier[$supplier_id] = [
            'name' => $receipt['supplier_name'],
            'receipts' => [],
            'added_count' => 0,
            'pending_count' => 0,
            'total_amount' => 0
        ];
    }
    $receipts_by_supplier[$supplier_id]['receipts'][] = $receipt;
    
    // Count status and calculate totals
    $payment_status = $receipt['payment_status'] ?? 'pending';
                    if ($receipt['status'] === 'yes') {
        $receipts_by_supplier[$supplier_id]['added_count']++;
    } else {
        $receipts_by_supplier[$supplier_id]['pending_count']++;
    }
    $receipts_by_supplier[$supplier_id]['total_amount'] += ($receipt['quantity'] * $receipt['unit_price']);
}

// Handle adding items to stock
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_to_stock'])) {
    $receipt_id = intval($_POST['receipt_id']);
    
    try {
        $pdo->beginTransaction();
        
        // Get the receipt data
        $stmt = $pdo->prepare("SELECT * FROM supplier_receipts WHERE id = ?");
        $stmt->execute([$receipt_id]);
        $receipt = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$receipt) {
            throw new Exception("Receipt not found");
        }

        if ($receipt['status'] !== 'pending' || !empty($receipt['order_id'])) {
            if (!empty($receipt['order_id'])) {
                throw new Exception("Items from orders are automatically added to stock and cannot be processed again.");
            }
            throw new Exception("This item has already been processed. Status: " . $receipt['status'] . ", Payment Status: " . $receipt['payment_status']);
        }

        // Check database structure
        $check_column = $pdo->query("SHOW COLUMNS FROM supplier_receipts LIKE 'status'");
        $column_info = $check_column->fetch(PDO::FETCH_ASSOC);
        error_log("Status column type: " . print_r($column_info, true));

        // First, check the current status value
        $check_stmt = $pdo->prepare("SELECT status FROM supplier_receipts WHERE id = ?");
        $check_stmt->execute([$receipt_id]);
        $current_status = $check_stmt->fetchColumn();
        error_log("Current status before update: " . ($current_status ?: 'null'));

        // Try direct update with explicit value
        $update_stmt = $pdo->prepare("
            UPDATE supplier_receipts 
            SET status = 'yes', 
                updated_at = NOW() 
            WHERE id = ? 
            AND status = 'pending'
        ");
        
        error_log("Attempting to update status for receipt ID: " . $receipt_id);
        $update_result = $update_stmt->execute([$receipt_id]);
        
        if (!$update_result) {
            $error_info = $update_stmt->errorInfo();
            error_log("Update failed. Error info: " . print_r($error_info, true));
            throw new Exception("Failed to update status: " . implode(", ", $error_info));
        }

        // Verify the update with a more detailed query
        $verify_stmt = $pdo->prepare("
            SELECT id, status, updated_at 
            FROM supplier_receipts 
            WHERE id = ?
        ");
        $verify_stmt->execute([$receipt_id]);
        $updated_receipt = $verify_stmt->fetch(PDO::FETCH_ASSOC);
        
        error_log("Verification query result: " . print_r($updated_receipt, true));
        
        if (!$updated_receipt) {
            throw new Exception("Could not verify the update - receipt not found after update");
        }

        // If status is still not 'yes', try one more time with a different approach
        if ($updated_receipt['status'] !== 'yes') {
            error_log("First update attempt failed, trying alternative approach");
            
            // Try updating with a different query
            $retry_stmt = $pdo->prepare("
                UPDATE supplier_receipts 
                SET status = 'yes' 
                WHERE id = ?
            ");
            $retry_stmt->execute([$receipt_id]);
            
            // Verify again
            $verify_stmt->execute([$receipt_id]);
            $updated_receipt = $verify_stmt->fetch(PDO::FETCH_ASSOC);
            error_log("Retry verification result: " . print_r($updated_receipt, true));
        }
        
        if ($updated_receipt['status'] !== 'yes') {
            throw new Exception("Status update verification failed. Current status: " . 
                ($updated_receipt['status'] ?: 'null') . 
                " (ID: " . $updated_receipt['id'] . 
                ", Updated at: " . $updated_receipt['updated_at'] . ")");
        }

        // Add serial numbers to product_serial_numbers if this receipt has them and they weren't added before
        if (!empty($receipt['serial_numbers'])) {
            $serials = json_decode($receipt['serial_numbers'], true);
            if (is_array($serials)) {
                // Check if serials are already in the system
                $placeholders = str_repeat('?,', count($serials) - 1) . '?';
                $check_stmt = $pdo->prepare("
                    SELECT COUNT(*) 
                    FROM product_serial_numbers 
                    WHERE serial_number IN ($placeholders) AND source_id = ?
                ");
                $check_stmt->execute(array_merge($serials, [$receipt_id]));
                
                // Only add if not already present
                if ($check_stmt->fetchColumn() == 0) {
                    $stmt = $pdo->prepare("
                        INSERT INTO product_serial_numbers 
                        (product_id, serial_number, source_type, source_id) 
                        VALUES (?, ?, 'supplier_receipt', ?)
                    ");
                    foreach ($serials as $serial) {
                        $stmt->execute([$receipt['product_id'], $serial, $receipt_id]);
                    }
                }
            }
        }

        // Update product stock
        $stmt = $pdo->prepare("
            UPDATE products 
            SET quantity = quantity + ?
            WHERE id = ?
        ");
        $stmt->execute([$receipt['quantity'], $receipt['product_id']]);

        // Log the stock adjustment
        $stmt = $pdo->prepare("
            INSERT INTO stock_adjustments 
            (product_id, quantity, reason, adjusted_by, adjusted_at, serial_numbers) 
            VALUES (?, ?, ?, ?, NOW(), ?)
        ");
        $stmt->execute([
            $receipt['product_id'],
            $receipt['quantity'],
            'Added from supplier receipt #' . $receipt_id,
            $_SESSION['user_id'],
            $receipt['serial_numbers']
        ]);

        $pdo->commit();
        $success = "Item added to stock successfully!";
        header('Location: receivesupplieritem.php?success=1');
        exit;
    } catch (Exception $e) {
        if ($pdo->inTransaction()) $pdo->rollBack();
        $error = "Error adding item to stock: " . $e->getMessage();
        error_log("Stock update error: " . $e->getMessage() . "\n" . $e->getTraceAsString());
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Receive Supplier Items - POS System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); font-family: 'Inter', sans-serif; }
        .card { border-radius: 16px; box-shadow: 0 2px 12px rgba(37,99,235,0.06); border: none; }
        .card-header { background: none; border-bottom: none; }
        .btn-primary, .btn-warning, .btn-info, .btn-danger { border-radius: 8px; font-weight: 600; }
        .form-label { font-weight: 600; color: #2563eb; }
        .modal-title { color: #2563eb; font-weight: 700; }
        .table { background: #fff; border-radius: 12px; overflow: hidden; }
        .table th, .table td { vertical-align: middle; }
        .supplier-header:hover {
            background-color: #f8f9fa;
        }
        .supplier-toggle {
            transition: transform 0.2s;
        }
        .supplier-details td {
            padding: 0 !important;
        }
        .supplier-details .table {
            margin-bottom: 0;
        }
        .btn-group-sm .btn {
            padding: 0.2rem 0.4rem;
            font-size: 0.75rem;
        }
        .btn-group-sm .btn i {
            font-size: 0.7rem;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                
                <!-- Main Content -->
                <div class="card mb-4">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h4 class="mb-0">Supplier Receipts</h4>
                        <div>
                            <button type="button" class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#createReceiptModal">
                                <i class="fas fa-plus"></i> Receive Supplier Items
                            </button>
                            <a href="suppliers.php" class="btn btn-secondary btn-sm">
                                <i class="fas fa-arrow-left"></i> Back
                            </a>
                        </div>
                    </div>
                    <div class="card-body">
                        <?php if ($success): ?>
                            <div class="alert alert-success"><?php echo $success; ?></div>
                        <?php endif; ?>
                        <?php if ($error): ?>
                            <div class="alert alert-danger"><?php echo $error; ?></div>
                        <?php endif; ?>

                        <!-- Filter Buttons -->
                        <div class="mb-3">
                            <button class="btn btn-outline-primary btn-sm me-2" id="filter-all" onclick="filterReceipts('all')">
                                <i class="fas fa-list"></i> All
                            </button>
                            <button class="btn btn-outline-success btn-sm me-2" id="filter-added" onclick="filterReceipts('added')">
                                <i class="fas fa-check"></i> Added to Stock
                            </button>
                            <button class="btn btn-outline-warning btn-sm" id="filter-pending" onclick="filterReceipts('pending')">
                                <i class="fas fa-clock"></i> Pending
                            </button>
                        </div>

                        <!-- Supplier Receipts by Supplier -->
                        <div id="supplier-receipts">
                            <?php foreach ($receipts_by_supplier as $supplier_id => $supplier_data): ?>
                                <div class="supplier-group mb-3">
                                    <!-- Supplier Header -->
                                    <div class="supplier-header p-3 border rounded bg-light" style="cursor: pointer;" data-supplier-id="<?php echo $supplier_id; ?>">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div>
                                                <h5 class="mb-0">
                                                    <i class="fas fa-chevron-right supplier-toggle me-2"></i>
                                                    <?php echo htmlspecialchars($supplier_data['name']); ?>
                                                </h5>
                                                <small class="text-muted">
                                                    <?php echo count($supplier_data['receipts']); ?> receipts
                                                </small>
                                            </div>
                                            <div class="text-end">
                                                <div class="d-flex gap-2 mb-1">
                                                    <span class="badge bg-success">Added: <?php echo $supplier_data['added_count']; ?></span>
                                                    <span class="badge bg-warning text-dark">Pending: <?php echo $supplier_data['pending_count']; ?></span>
                                                </div>
                                                <strong>Total: <?php echo number_format($supplier_data['total_amount'], 2); ?></strong>
                                            </div>
                                        </div>
                                    </div>

                                    <!-- Supplier Receipts Table -->
                                    <div class="supplier-receipts" data-supplier-id="<?php echo $supplier_id; ?>" style="display: none;">
                                        <div class="table-responsive mt-2">
                                            <table class="table table-sm table-bordered">
                                <thead class="table-light">
                                    <tr>
                                        <th>Date</th>
                                        <th>Batch Number</th>
                                                                <th>Product</th>
                                                                <th>Qty</th>
                                                                <th>Price</th>
                                                                <th>Total</th>
                                                        <th>Serial Numbers</th>
                                                                <th>Status</th>
                                                        <th>Created By</th>
                                                                <th>Actions</th>
                                                            </tr>
                                                        </thead>
                                                        <tbody>
                                                    <?php foreach ($supplier_data['receipts'] as $receipt): ?>
                                                        <?php
                                                            $payment_status = $receipt['payment_status'] ?? 'pending';
                                                            $is_added = ($receipt['status'] === 'yes');
                                                            $row_class = $is_added ? 'added-to-stock' : 'pending';
                                                        ?>
                                                        <tr class="receipt-row <?php echo $row_class; ?>" data-status="<?php echo $is_added ? 'added' : 'pending'; ?>">
                                                            <td><?php echo date('Y-m-d H:i', strtotime($receipt['created_at'])); ?></td>
                                                            <td><?php echo htmlspecialchars($receipt['batch_number']); ?></td>
                                                            <td><?php echo htmlspecialchars($receipt['product_name']); ?></td>
                                                            <td><?php echo number_format($receipt['quantity']); ?></td>
                                                            <td><?php echo number_format($receipt['unit_price'], 2); ?></td>
                                                            <td><?php echo number_format($receipt['quantity'] * $receipt['unit_price'], 2); ?></td>
                                                                    <td>
                                                                        <?php 
                                                                if (!empty($receipt['serial_numbers'])) {
                                                                    $serials = json_decode($receipt['serial_numbers'], true);
                                                                    echo is_array($serials) ? htmlspecialchars(implode(', ', $serials)) : htmlspecialchars($receipt['serial_numbers']);
                                                                        } else { echo '-'; }
                                                                        ?>
                                                                    </td>
                                                                    <td>
                                                                <?php if ($is_added): ?>
                                                                    <span class="badge bg-success">Added to Stock</span>
                                                                <?php elseif ($payment_status === 'paid'): ?>
                                                                    <span class="badge bg-info">Paid</span>
                                                                <?php elseif ($payment_status === 'credit'): ?>
                                                                    <span class="badge bg-warning text-dark">On Credit</span>
                                                                        <?php else: ?>
                                                                    <span class="badge bg-secondary">Pending</span>
                                                                        <?php endif; ?>
                                                                    </td>
                                                            <td><?php echo htmlspecialchars($receipt['created_by_name']); ?></td>
                                                                    <td>
                                                                <?php 
                                                                $order_id = $receipt['order_id'] ?? null;
                                                                if (!$is_added && empty($order_id)): ?>
                                                                            <form method="post" style="display: inline;">
                                                                        <input type="hidden" name="receipt_id" value="<?php echo $receipt['id']; ?>">
                                                                                <button type="submit" name="add_to_stock" class="btn btn-success btn-sm" title="Add Item to Stock">
                                                                                    <i class="fas fa-check"></i>
                                                                                </button>
                                                                            </form>
                                                                <?php elseif ($is_added || !empty($order_id)): ?>
                                                                    <span class="badge bg-info">Stock Added</span>
                                                                        <?php endif; ?>
                                                                    </td>
                                                                </tr>
                                                            <?php endforeach; ?>
                                                        </tbody>
                                                    </table>
                                                </div>
                                    </div>
                                </div>
                                    <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Create Receipt Modal -->
    <div class="modal fade" id="createReceiptModal" tabindex="-1" aria-labelledby="createReceiptModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="createReceiptModalLabel">Create New Receipt</h5>
                    <button type="button" class="btn-close" onclick="window.location.href='receivesupplieritem.php'" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <form method="post" class="needs-validation" novalidate>
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label class="form-label">Receipt Type *</label>
                                <select class="form-select form-select-sm" id="receipt_type" name="receipt_type" required onchange="handleReceiptTypeChange()">
                                    <option value="">Select Receipt Type</option>
                                    <option value="manual">Manual Entry</option>
                                    <option value="from_order">Receive from Order</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label class="form-label">Supplier *</label>
                                <select class="form-select form-select-sm" id="supplier_select" name="supplier_id" required onchange="handleSupplierChange()">
                                    <option value="">Select Supplier</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?php echo $supplier['id']; ?>">
                                            <?php echo htmlspecialchars($supplier['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-12" id="order_selection_section" style="display: none;">
                                <label class="form-label">Select Order *</label>
                                <select class="form-select form-select-sm" id="order_select" name="order_id" onchange="handleOrderChange()">
                                    <option value="">Select Order</option>
                                </select>
                                <small class="text-muted">Only approved orders are shown</small>
                            </div>
                            <div class="col-12">
                                <label class="form-label">Items *</label>
                                <table class="table table-bordered" id="itemsTable">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Quantity</th>
                                            <th>Unit Price</th>
                                            <th>Batch Number</th>
                                            <th class="serial-column">Serial Numbers</th>
                                            <th></th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr class="item-row">
                                            <td>
                                                <input type="hidden" class="order-item-id" name="order_item_id[]" value="">
                                                <select class="form-select form-select-sm product-select" name="product_id[]" required onchange="handleProductChange(this)">
                                                    <option value="">Select Product</option>
                                                    <?php foreach ($products as $product): ?>
                                                        <option value="<?php echo $product['id']; ?>" 
                                                                data-require-serials="<?php echo $product['require_serials']; ?>">
                                                            <?php echo htmlspecialchars($product['name']); ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </td>
                                            <td><input type="number" class="form-control form-control-sm quantity-input" name="quantity[]" min="1" required onchange="handleQuantityChange(this)"></td>
                                            <td><input type="number" class="form-control form-control-sm" name="unit_price[]" step="0.01" min="0.01" required></td>
                                            <td><input type="text" class="form-control form-control-sm batch-number" name="batch_number[]" readonly required></td>
                                            <td class="serial-column">
                                                <div class="serial-numbers-container">
                                                    <input type="text" class="form-control form-control-sm serial-numbers-input" name="serial_numbers[]" readonly>
                                                    <div class="btn-group btn-group-sm mt-1 serial-buttons" style="display: none;">
                                                        <button type="button" class="btn btn-primary btn-sm generate-serial-btn">
                                                            <i class="fas fa-magic"></i> Gen
                                                        </button>
                                                        <button type="button" class="btn btn-secondary btn-sm manual-serial-btn">
                                                            <i class="fas fa-edit"></i> Manual
                                                        </button>
                                                    </div>
                                                    <small class="text-muted serial-requirement-text" style="display: none;"></small>
                                                </div>
                                            </td>
                                            <td><button type="button" class="btn btn-danger btn-sm remove-item">&times;</button></td>
                                        </tr>
                                    </tbody>
                                </table>
                                <button type="button" class="btn btn-success btn-sm" id="addItemRow"><i class="fas fa-plus"></i> Add Another Item</button>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary btn-sm" onclick="window.location.href='receivesupplieritem.php'">Cancel</button>
                            <button type="submit" name="receive_item" class="btn btn-primary btn-sm">
                                <i class="fas fa-save"></i> Create Receipt
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Payment Modal -->
    <div class="modal fade" id="paymentModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Payment Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="paymentForm" method="post">
                        <input type="hidden" name="receipt_id" id="payment_receipt_id">
                        <div class="mb-3">
                            <label class="form-label">Payment Status</label>
                            <select class="form-select" name="payment_status" id="payment_status" required onchange="handlePaymentStatusChange()">
                                <option value="">Select Payment Status</option>
                                <option value="paid">Paid</option>
                                <option value="credit">On Credit</option>
                            </select>
                        </div>
                        <div id="paymentMethodSection" style="display: none;">
                            <div class="mb-3">
                                <label class="form-label">Payment Method</label>
                                <select class="form-select" name="payment_method" id="payment_method">
                                    <option value="">Select Payment Method</option>
                                    <option value="cash">Cash</option>
                                    <option value="mpesa">M-PESA</option>
                                </select>
                            </div>
                            <div id="mpesaSection" style="display: none;">
                                <div class="mb-3">
                                    <label class="form-label">M-PESA Transaction Code</label>
                                    <input type="text" class="form-control" name="mpesa_code" id="mpesa_code">
                                </div>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Payment Amount</label>
                            <input type="number" class="form-control" name="payment_amount" id="payment_amount" min="0.01" step="0.01" required>
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" onclick="submitPayment()">Save Payment</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Serial Numbers Generator Modal -->
    <div class="modal fade" id="serialNumbersGeneratorModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Generate Serial Numbers</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Prefix (optional)</label>
                        <input type="text" class="form-control" id="serialPrefix" placeholder="e.g., SN-">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Starting Number</label>
                        <input type="number" class="form-control" id="serialStart" value="1" min="1">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Number of Digits</label>
                        <input type="number" class="form-control" id="serialDigits" value="5" min="1" max="10">
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Preview</label>
                        <div id="serialPreview" class="form-control" style="height: auto; min-height: 100px;"></div>
                    </div>
                    <div class="alert alert-danger" id="generateSerialError" style="display: none;"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="applySerialNumbers">Apply</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Manual Serial Numbers Modal -->
    <div class="modal fade" id="manualSerialModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Enter Serial Numbers</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">Enter Serial Numbers (comma-separated)</label>
                        <textarea class="form-control" id="manualSerialInput" rows="5" placeholder="Enter serial numbers separated by commas"></textarea>
                    </div>
                    <div class="alert alert-danger" id="serialError" style="display: none;"></div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="applyManualSerial">Apply</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Edit Item Modal -->
    <div class="modal fade" id="editItemModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Supplier Item</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <form id="editItemForm">
                        <input type="hidden" id="editRowIndex">
                        <div class="mb-3">
                            <label class="form-label">Product</label>
                            <input type="text" class="form-control" id="editProduct" readonly>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Quantity</label>
                            <input type="number" class="form-control" id="editQuantity" min="1" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Unit Price</label>
                            <input type="number" class="form-control" id="editUnitPrice" step="0.01" min="0.01" required>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Batch Number</label>
                            <input type="text" class="form-control" id="editBatchNumber" readonly>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Serial Numbers</label>
                            <input type="text" class="form-control" id="editSerialNumbers">
                        </div>
                    </form>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="button" class="btn btn-primary" id="saveEditItem">Save</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Data from PHP
        const approvedOrders = <?php echo json_encode($approved_orders); ?>;
        const orderItems = <?php echo json_encode($order_items); ?>;
        const products = <?php echo json_encode($products); ?>;
        
        // Handle receipt type change
        function handleReceiptTypeChange() {
            const receiptType = document.getElementById('receipt_type').value;
            const orderSection = document.getElementById('order_selection_section');
            const addItemButton = document.getElementById('addItemRow');
            const itemsTable = document.getElementById('itemsTable');
            
            if (receiptType === 'from_order') {
                orderSection.style.display = 'block';
                addItemButton.style.display = 'none';
                document.getElementById('order_select').required = true;
                
                // Clear existing items except header
                const tbody = itemsTable.querySelector('tbody');
                tbody.innerHTML = '';
            } else if (receiptType === 'manual') {
                orderSection.style.display = 'none';
                addItemButton.style.display = 'inline-block';
                document.getElementById('order_select').required = false;
                
                // Reset to manual entry mode - add one empty row
                resetToManualEntry();
                
                // Set manual batch number for all rows
                setManualBatchNumberForAllRows();
            } else {
                orderSection.style.display = 'none';
                addItemButton.style.display = 'none';
                document.getElementById('order_select').required = false;
            }
        }
        
        // Handle supplier change
        function handleSupplierChange() {
            const supplierId = document.getElementById('supplier_select').value;
            const receiptType = document.getElementById('receipt_type').value;
            const orderSelect = document.getElementById('order_select');
            
            // Clear order selection
            orderSelect.innerHTML = '<option value="">Select Order</option>';
            
            if (receiptType === 'from_order' && supplierId) {
                // Filter orders by supplier
                const supplierOrders = approvedOrders.filter(order => order.supplier_id == supplierId);
                
                supplierOrders.forEach(order => {
                    const option = document.createElement('option');
                    option.value = order.order_id;
                    option.textContent = `Order #${order.order_id} - ${order.order_date}`;
                    orderSelect.appendChild(option);
                });
                
                if (supplierOrders.length === 0) {
                    const option = document.createElement('option');
                    option.value = '';
                    option.textContent = 'No approved orders for this supplier';
                    option.disabled = true;
                    orderSelect.appendChild(option);
                }
            }
        }
        
        // Handle order change
        function handleOrderChange() {
            const orderId = document.getElementById('order_select').value;
            const itemsTable = document.getElementById('itemsTable');
            const tbody = itemsTable.querySelector('tbody');
            
            // Clear existing items
            tbody.innerHTML = '';
            
            if (orderId && orderItems[orderId]) {
                const items = orderItems[orderId];
                
                items.forEach(item => {
                    const row = createOrderItemRow(item);
                    tbody.appendChild(row);
                });
                
                // Update batch numbers for order items
                updateBatchNumbersForOrder();
            }
        }
        
        // Create a row for order items
        function createOrderItemRow(item) {
            const row = document.createElement('tr');
            row.className = 'item-row';
            
            const remainingQty = item.quantity - (item.received_quantity || 0);
            const requireSerials = parseInt(item.require_serials) === 1;
            
            row.innerHTML = `
                <td>
                    <input type="hidden" class="order-item-id" name="order_item_id[]" value="${item.order_item_id}">
                    <input type="hidden" name="product_id[]" value="${item.product_id}">
                    <div class="form-control form-control-sm" style="background: #f8f9fa;">
                        ${item.product_name}
                        ${requireSerials ? '<small class="text-danger d-block">Serial numbers required</small>' : ''}
                        </div>
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm quantity-input" 
                           name="quantity[]" min="1" max="${remainingQty}" value="${remainingQty}" 
                           required onchange="handleQuantityChange(this)" 
                           data-max-qty="${remainingQty}">
                    <small class="text-muted">Max: ${remainingQty}</small>
                </td>
                <td>
                    <input type="number" class="form-control form-control-sm" 
                           name="unit_price[]" step="0.01" min="0.01" required>
                </td>
                <td>
                    <input type="text" class="form-control form-control-sm batch-number" 
                           name="batch_number[]" readonly required>
                </td>
                <td class="serial-column">
                    <div class="serial-numbers-container">
                        <input type="text" class="form-control form-control-sm serial-numbers-input" 
                               name="serial_numbers[]" readonly ${requireSerials ? 'required' : ''}>
                        <div class="btn-group btn-group-sm mt-1 serial-buttons" style="${requireSerials ? 'display: block;' : 'display: none;'}">
                            <button type="button" class="btn btn-secondary btn-sm manual-serial-btn">
                                <i class="fas fa-edit"></i> Manual
                            </button>
                        </div>
                        <small class="text-muted serial-requirement-text" style="${requireSerials ? 'display: block;' : 'display: none;'}">
                            ${requireSerials ? 'Serial numbers required (will match quantity)' : ''}
                        </small>
                    </div>
                </td>
                <td>
                    <button type="button" class="btn btn-danger btn-sm remove-item" onclick="removeOrderItem(this)" disabled>
                        &times;
                    </button>
                </td>
            `;
            
            return row;
        }
        
        // Handle product change (for manual entry)
        function handleProductChange(selectElement) {
            const row = selectElement.closest('tr');
            const selectedOption = selectElement.options[selectElement.selectedIndex];
            const requireSerials = selectedOption.dataset.requireSerials === '1';
            
            // Update serial number requirements
            updateSerialRequirements(row, requireSerials);
        }
        
        // Handle quantity change
        function handleQuantityChange(inputElement) {
            const row = inputElement.closest('tr');
            const quantity = parseInt(inputElement.value) || 0;
            const maxQty = parseInt(inputElement.dataset.maxQty);
            
            // Validate against max quantity for order items
            if (maxQty && quantity > maxQty) {
                inputElement.value = maxQty;
                alert(`Quantity cannot exceed ${maxQty} (remaining to receive)`);
                return;
            }
            
            // Update serial number requirements
            const productSelect = row.querySelector('.product-select');
            if (productSelect) {
                const selectedOption = productSelect.options[productSelect.selectedIndex];
                const requireSerials = selectedOption.dataset.requireSerials === '1';
                updateSerialRequirements(row, requireSerials, quantity);
            }
        }
        
        // Update serial number requirements for a row
        function updateSerialRequirements(row, requireSerials, quantity = null) {
            const serialButtons = row.querySelector('.serial-buttons');
            const serialInput = row.querySelector('.serial-numbers-input');
            const requirementText = row.querySelector('.serial-requirement-text');
            
            if (requireSerials) {
                serialButtons.style.display = 'block';
                serialInput.required = true;
                requirementText.style.display = 'block';
                
                if (quantity) {
                    requirementText.textContent = `Serial numbers required (${quantity} needed)`;
            } else {
                    requirementText.textContent = 'Serial numbers required (will match quantity)';
                }
            } else {
                serialButtons.style.display = 'none';
                serialInput.required = false;
                requirementText.style.display = 'none';
                serialInput.value = '';
            }
        }
        
        // Reset to manual entry mode
        function resetToManualEntry() {
            const tbody = document.querySelector('#itemsTable tbody');
            tbody.innerHTML = `
                <tr class="item-row">
                    <td>
                        <input type="hidden" class="order-item-id" name="order_item_id[]" value="">
                        <select class="form-select form-select-sm product-select" name="product_id[]" required onchange="handleProductChange(this)">
                            <option value="">Select Product</option>
                            ${products.map(product => 
                                `<option value="${product.id}" data-require-serials="${product.require_serials}">
                                    ${product.name}
                                </option>`
                            ).join('')}
                        </select>
                    </td>
                    <td><input type="number" class="form-control form-control-sm quantity-input" name="quantity[]" min="1" required onchange="handleQuantityChange(this)"></td>
                    <td><input type="number" class="form-control form-control-sm" name="unit_price[]" step="0.01" min="0.01" required></td>
                    <td><input type="text" class="form-control form-control-sm batch-number" name="batch_number[]" readonly required></td>
                    <td class="serial-column">
                        <div class="serial-numbers-container">
                            <input type="text" class="form-control form-control-sm serial-numbers-input" name="serial_numbers[]" readonly>
                            <div class="btn-group btn-group-sm mt-1 serial-buttons" style="display: none;">
                                <button type="button" class="btn btn-primary btn-sm generate-serial-btn">
                                    <i class="fas fa-magic"></i> Gen
                                </button>
                                <button type="button" class="btn btn-secondary btn-sm manual-serial-btn">
                                    <i class="fas fa-edit"></i> Manual
                                </button>
                            </div>
                            <small class="text-muted serial-requirement-text" style="display: none;"></small>
                        </div>
                    </td>
                    <td><button type="button" class="btn btn-danger btn-sm remove-item">&times;</button></td>
                </tr>
            `;
        }
        
        // Remove order item (for display only - they should be disabled)
        function removeOrderItem(button) {
            // Order items should not be removable
            alert('Order items cannot be removed. Adjust the quantity to 0 if not receiving this item.');
        }
    </script>
    <script>


        function handlePaymentStatusChange() {
            const statusElement = document.getElementById('payment_status');
            const methodSection = document.getElementById('paymentMethodSection');
            const mpesaSection = document.getElementById('mpesaSection');
            const paymentAmountInput = document.getElementById('payment_amount');
            
            // Check if required elements exist
            if (!statusElement || !methodSection || !mpesaSection || !paymentAmountInput) {
                console.warn('Required elements not found for payment status change');
                return;
            }
            
            const status = statusElement.value;
            
            if (status === 'paid') {
                methodSection.style.display = 'block';
                paymentAmountInput.disabled = false;
                paymentAmountInput.value = '';
                paymentAmountInput.required = true;
            } else if (status === 'credit') {
                methodSection.style.display = 'none';
                mpesaSection.style.display = 'none';
                paymentAmountInput.disabled = true;
                paymentAmountInput.value = 0;
                paymentAmountInput.required = false;
            } else {
                methodSection.style.display = 'none';
                mpesaSection.style.display = 'none';
                paymentAmountInput.disabled = false;
                paymentAmountInput.value = '';
                paymentAmountInput.required = true;
            }
        }

        const paymentMethodElement = document.getElementById('payment_method');
        if (paymentMethodElement) {
            paymentMethodElement.addEventListener('change', function() {
            const mpesaSection = document.getElementById('mpesaSection');
                if (mpesaSection) {
            mpesaSection.style.display = this.value === 'mpesa' ? 'block' : 'none';
                }
        });
        }

        function submitPayment() {
            const form = document.getElementById('paymentForm');
            const receiptIdElement = document.getElementById('payment_receipt_id');
            const statusElement = document.getElementById('payment_status');
            const amountElement = document.getElementById('payment_amount');
            
            // Check if required elements exist
            if (!form || !receiptIdElement || !statusElement || !amountElement) {
                alert('Payment form not properly initialized. Please refresh the page and try again.');
                return;
            }
            
            const formData = new FormData(form);
            const receiptId = receiptIdElement.value;
            const status = statusElement.value;
            let amount;
            
            if (status === 'credit') {
                amount = 0;
            } else {
                amount = parseFloat(amountElement.value);
                if (!amount || amount <= 0) {
                    alert('Please enter a valid payment amount.');
                    return;
                }
            }

            const paymentDistribution = [{
                item_id: receiptId,
                amount: amount
            }];
            formData.append('payment_distribution', JSON.stringify(paymentDistribution));

            fetch('process_payment.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    alert('Payment recorded successfully!');
                    window.location.href = 'receivesupplieritem.php?success=1';
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while processing the payment.');
            });
        }

        // Enhanced form validation
        (function () {
            'use strict'
            var forms = document.querySelectorAll('.needs-validation')
            Array.prototype.slice.call(forms).forEach(function (form) {
                form.addEventListener('submit', function (event) {
                    let isValid = form.checkValidity();
                    
                    // Additional validation for serial numbers
                    const serialInputs = form.querySelectorAll('.serial-numbers-input[required]');
                    serialInputs.forEach(input => {
                        const row = input.closest('tr');
                        const quantityInput = row.querySelector('.quantity-input');
                        const quantity = parseInt(quantityInput.value) || 0;
                        
                        if (input.required && input.value.trim()) {
                            const serials = input.value.split(',').map(s => s.trim()).filter(s => s);
                            if (serials.length !== quantity) {
                                input.setCustomValidity(`Please enter exactly ${quantity} serial numbers`);
                                isValid = false;
                            } else if (new Set(serials).size !== serials.length) {
                                input.setCustomValidity('Serial numbers must be unique');
                                isValid = false;
                            } else {
                                input.setCustomValidity('');
                            }
                        }
                    });
                    
                    // Check if receiving from order and validate order selection
                    const receiptType = document.getElementById('receipt_type').value;
                    if (receiptType === 'from_order') {
                        const orderSelect = document.getElementById('order_select');
                        if (!orderSelect.value) {
                            orderSelect.setCustomValidity('Please select an order');
                            isValid = false;
                        } else {
                            orderSelect.setCustomValidity('');
                        }
                    }
                    
                    if (!isValid) {
                        event.preventDefault();
                        event.stopPropagation();
                    }
                    form.classList.add('was-validated');
                }, false);
            });
        })()

        // Reset form when modal is closed
        const createReceiptModal = document.getElementById('createReceiptModal');
        if (createReceiptModal) {
            createReceiptModal.addEventListener('hidden.bs.modal', function () {
            const form = this.querySelector('form');
                if (form) {
            form.reset();
            form.classList.remove('was-validated');
                }
                
                // Reset visibility of sections
                const orderSection = document.getElementById('order_selection_section');
                const addItemButton = document.getElementById('addItemRow');
                if (orderSection) orderSection.style.display = 'none';
                if (addItemButton) addItemButton.style.display = 'none';
                
                // Clear the items table
                const tbody = document.querySelector('#itemsTable tbody');
                if (tbody) tbody.innerHTML = '';
                
                // Reset validation
                const orderSelect = document.getElementById('order_select');
                if (orderSelect) orderSelect.setCustomValidity('');
            });
        }

        document.querySelectorAll('.supplier-header').forEach(header => {
            header.addEventListener('click', function() {
                const supplierId = this.dataset.supplierId;
                const details = document.querySelector(`.supplier-details[data-supplier-id="${supplierId}"]`);
                const toggle = this.querySelector('.supplier-toggle');
                
                if (details.style.display === 'none') {
                    details.style.display = 'table-row';
                    toggle.classList.remove('fa-chevron-right');
                    toggle.classList.add('fa-chevron-down');
                } else {
                    details.style.display = 'none';
                    toggle.classList.remove('fa-chevron-down');
                    toggle.classList.add('fa-chevron-right');
                }
            });
        });

        function showPaymentModal(receiptId) {
            document.getElementById('payment_receipt_id').value = receiptId;
            new bootstrap.Modal(document.getElementById('paymentModal')).show();
        }

        // When creating a new receipt, pre-select the supplier
        document.querySelectorAll('[data-supplier-id]').forEach(button => {
            button.addEventListener('click', function() {
                const supplierId = this.dataset.supplierId;
                document.querySelector('select[name="supplier_id"]').value = supplierId;
            });
        });

        function viewSerialNumbers(serialNumbers) {
            const modal = new bootstrap.Modal(document.getElementById('serialNumbersModal'));
            const tbody = document.getElementById('serialNumbersList');
            tbody.innerHTML = '';
            
            // Parse serial numbers if it's a JSON string
            if (typeof serialNumbers === 'string') {
                try {
                    serialNumbers = JSON.parse(serialNumbers);
                } catch (e) {
                    serialNumbers = [serialNumbers];
                }
            }
            
            // If it's not an array, make it one
            if (!Array.isArray(serialNumbers)) {
                serialNumbers = [serialNumbers];
            }
            
            // Add each serial number to the table
            serialNumbers.forEach((serial, index) => {
                const row = document.createElement('tr');
                row.innerHTML = `
                    <td>${index + 1}</td>
                    <td>${serial}</td>
                `;
                tbody.appendChild(row);
            });
            
            modal.show();
        }

        function approveItem(receiptId) {
            if (confirm('Are you sure you want to approve this item?')) {
                updateItemStatus(receiptId, 'approved');
            }
        }

        function rejectItem(receiptId) {
            if (confirm('Are you sure you want to reject this item?')) {
                updateItemStatus(receiptId, 'rejected');
            }
        }

        function approveAllItems(supplierId) {
            if (confirm('Are you sure you want to approve all pending items for this supplier?')) {
                fetch('update_supplier_receipts.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `action=approve_all&supplier_id=${supplierId}`
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('An error occurred while updating the items.');
                });
            }
        }

        function updateItemStatus(receiptId, status) {
            fetch('update_supplier_receipts.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `action=update_status&receipt_id=${receiptId}&status=${status}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    location.reload();
                } else {
                    alert('Error: ' + data.message);
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred while updating the item status.');
            });
        }

        // Consolidated addItemRow event listener
        document.getElementById('addItemRow').addEventListener('click', function() {
            const table = document.getElementById('itemsTable').querySelector('tbody');
            const newRow = table.rows[0].cloneNode(true);
            
            // Clear all inputs
            newRow.querySelectorAll('input, select').forEach(input => {
                input.value = '';
            });
            newRow.querySelector('select').selectedIndex = 0;
            
            // Add event listeners to new row
            const quantityInput = newRow.querySelector('.quantity-input');
            if (quantityInput) {
                quantityInput.addEventListener('change', function() {
                handleQuantityChange(this);
            });
            }
            
            const generateBtn = newRow.querySelector('.generate-serial-btn');
            if (generateBtn) {
                generateBtn.addEventListener('click', function() {
                showSerialGenerator(this.closest('tr'));
            });
            }
            
            const manualBtn = newRow.querySelector('.manual-serial-btn');
            if (manualBtn) {
                manualBtn.addEventListener('click', function() {
                showManualSerialInput(this.closest('tr'));
            });
            }
            
            // Add event listener to new product select
            const productSelect = newRow.querySelector('.product-select');
            if (productSelect) {
                productSelect.addEventListener('change', function() {
                    handleProductChange(this);
                });
            }
            
            table.appendChild(newRow);
            
            // Set the appropriate batch number for the new row
            const receiptType = document.getElementById('receipt_type').value;
            if (receiptType === 'manual') {
                setManualBatchNumberForAllRows();
            } else {
                setBatchNumberForAllRows();
            }
        });

        document.getElementById('itemsTable').addEventListener('click', function(e) {
            if (e.target.classList.contains('remove-item')) {
                const row = e.target.closest('tr');
                if (row.parentNode.rows.length > 1) row.remove();
            }
        });

        // Store the current batch number for the modal
        let currentBatchNumber = '';

        // Function to fetch and set the next available batch number for the whole receipt
        async function fetchAndSetBatchNumber() {
            try {
                const response = await fetch('get_next_batch_number.php');
                const data = await response.json();
                if (data.success) {
                    currentBatchNumber = data.next_batch_number;
                } else {
                    currentBatchNumber = 'ERROR';
                }
            } catch (e) {
                currentBatchNumber = 'ERROR';
            }
        }

        // Function to set the batch number for all item rows
        function setBatchNumberForAllRows() {
            document.querySelectorAll('.batch-number').forEach(input => {
                input.value = currentBatchNumber;
                input.readOnly = true;
            });
        }
        
        // Function to set manual batch number for all item rows
        function setManualBatchNumberForAllRows() {
            document.querySelectorAll('.batch-number').forEach(input => {
                input.value = nextManualBatchNumber;
                input.readOnly = true;
            });
        }

        // When the modal is shown, fetch and set the batch number for all rows
        document.getElementById('createReceiptModal').addEventListener('shown.bs.modal', async function () {
            await fetchAndSetBatchNumber();
            setBatchNumberForAllRows();
        });

        // Update batch numbers when order changes
        function updateBatchNumbersForOrder() {
            const orderId = document.getElementById('order_select').value;
            if (orderId) {
                // For orders, use a batch number that includes the order ID
                const orderBatchNumber = `ORD${orderId}_${currentBatchNumber}`;
                document.querySelectorAll('.batch-number').forEach(input => {
                    input.value = orderBatchNumber;
                });
            } else {
                setBatchNumberForAllRows();
            }
        }



        // On modal close, reset the batch number variable
        document.getElementById('createReceiptModal').addEventListener('hidden.bs.modal', function () {
            currentBatchNumber = '';
        });

        function showSerialGenerator(row) {
            currentRow = row;
            const quantity = parseInt(row.querySelector('.quantity-input').value) || 0;
            const modal = new bootstrap.Modal(document.getElementById('serialNumbersGeneratorModal'));
            
            // Reset form
            document.getElementById('serialPrefix').value = '';
            document.getElementById('serialStart').value = '1';
            document.getElementById('serialDigits').value = '5';
            document.getElementById('generateSerialError').style.display = 'none';
            
            // Update preview when inputs change
            document.getElementById('serialPrefix').addEventListener('input', updatePreview);
            document.getElementById('serialStart').addEventListener('input', updatePreview);
            document.getElementById('serialDigits').addEventListener('input', updatePreview);
            
            updatePreview();
            modal.show();
        }

        function updatePreview() {
            const prefix = document.getElementById('serialPrefix').value;
            const start = parseInt(document.getElementById('serialStart').value) || 1;
            const digits = parseInt(document.getElementById('serialDigits').value) || 5;
            const quantity = parseInt(currentRow.querySelector('.quantity-input').value) || 0;
            
            // Get the last used serial number
            fetch('get_last_serial.php?prefix=' + encodeURIComponent(prefix))
                .then(response => response.json())
                .then(data => {
                    let startNumber = data.lastNumber + 1;
                    let preview = '';
                    for (let i = 0; i < quantity; i++) {
                        const number = String(startNumber + i).padStart(digits, '0');
                        preview += prefix + number + '\n';
                    }
                    document.getElementById('serialPreview').textContent = preview;
                })
                .catch(error => {
                    console.error('Error:', error);
                    // Fallback to original behavior if fetch fails
                    let preview = '';
                    for (let i = 0; i < quantity; i++) {
                        const number = String(start + i).padStart(digits, '0');
                        preview += prefix + number + '\n';
                    }
                    document.getElementById('serialPreview').textContent = preview;
                });
        }

        document.getElementById('applySerialNumbers').addEventListener('click', async function() {
            const preview = document.getElementById('serialPreview').textContent;
            const serials = preview.trim().split('\n');
            const errorDiv = document.getElementById('generateSerialError');
            
            try {
                const error = await validateSerialNumbers(serials.join(','));
                if (error) {
                    errorDiv.textContent = error;
                    errorDiv.style.display = 'block';
                    return;
                }
                
                // Apply the serial numbers
                const serialInput = currentRow.querySelector('.serial-numbers-input');
                serialInput.value = serials.join(', ');
                
                // Close the modal
                bootstrap.Modal.getInstance(document.getElementById('serialNumbersGeneratorModal')).hide();
            } catch (error) {
                errorDiv.textContent = 'An error occurred while validating serial numbers.';
                errorDiv.style.display = 'block';
            }
        });

        // Add these new functions for manual serial number handling
        function showManualSerialInput(row) {
            currentRow = row;
            const modal = new bootstrap.Modal(document.getElementById('manualSerialModal'));
            document.getElementById('manualSerialInput').value = '';
            document.getElementById('serialError').style.display = 'none';
            modal.show();
        }

        function validateSerialNumbers(serials) {
            // Remove any whitespace and split by comma
            const serialArray = serials.split(',').map(s => s.trim()).filter(s => s);
            
            // Check for empty input
            if (serialArray.length === 0) {
                return 'Please enter at least one serial number.';
            }
            
            // Check for duplicates
            if (serialArray.length !== new Set(serialArray).size) {
                return 'Duplicate serial numbers are not allowed.';
            }
            
            // Check against database (you'll need to implement this endpoint)
            return fetch('check_serial_numbers.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: 'serial_numbers=' + encodeURIComponent(JSON.stringify(serialArray))
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (!data.success) {
                    return data.message;
                }
                return null;
            })
            .catch(error => {
                console.error('Error validating serial numbers:', error);
                // If it's an SSL certificate error, try with a different approach
                if (error.message.includes('CERT_AUTHORITY_INVALID') || error.message.includes('SSL')) {
                    return 'Network error: Please check your connection and try again.';
                }
                return 'An error occurred while validating serial numbers. Please try again.';
            });
        }

        document.getElementById('applyManualSerial').addEventListener('click', async function() {
            const serialInput = document.getElementById('manualSerialInput');
            const errorDiv = document.getElementById('serialError');
            const serials = serialInput.value;
            const serialArray = serials.split(',').map(s => s.trim()).filter(s => s);

            // Get the quantity for the current row
            const quantity = parseInt(currentRow.querySelector('.quantity-input').value) || 0;

            // Check if the number of serials matches the quantity
            if (serialArray.length !== quantity) {
                errorDiv.textContent = 'Serial number are not equal to Quantity';
                errorDiv.style.display = 'block';
                return;
            }

            try {
                const error = await validateSerialNumbers(serials);
                if (error) {
                    errorDiv.textContent = error;
                    errorDiv.style.display = 'block';
                    return;
                }

                // Apply the serial numbers
                const serialInputField = currentRow.querySelector('.serial-numbers-input');
                serialInputField.value = serials;

                // Close the modal
                bootstrap.Modal.getInstance(document.getElementById('manualSerialModal')).hide();
            } catch (error) {
                errorDiv.textContent = 'An error occurred while validating serial numbers.';
                errorDiv.style.display = 'block';
            }
        });

        // Add event listeners for manual buttons
        document.querySelectorAll('.manual-serial-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                showManualSerialInput(this.closest('tr'));
            });
        });

        // Use event delegation for dynamically created elements
        document.addEventListener('click', function(e) {
            // Handle generate serial button clicks
            if (e.target.classList.contains('generate-serial-btn') || e.target.closest('.generate-serial-btn')) {
                const btn = e.target.classList.contains('generate-serial-btn') ? e.target : e.target.closest('.generate-serial-btn');
                showSerialGenerator(btn.closest('tr'));
            }
            
            // Handle manual serial button clicks
            if (e.target.classList.contains('manual-serial-btn') || e.target.closest('.manual-serial-btn')) {
                const btn = e.target.classList.contains('manual-serial-btn') ? e.target : e.target.closest('.manual-serial-btn');
                showManualSerialInput(btn.closest('tr'));
            }
        });

        // Add JS to open the modal and fill it with the row's data
        document.querySelectorAll('.edit-item-btn').forEach(function(btn) {
            btn.addEventListener('click', function() {
                var row = btn.closest('tr');
                var tds = row.querySelectorAll('td');
                // Adjust indices as per your table structure
                document.getElementById('editProduct').value = tds[2].innerText.trim();
                document.getElementById('editQuantity').value = tds[4].innerText.trim();
                document.getElementById('editUnitPrice').value = tds[5].innerText.trim();
                document.getElementById('editBatchNumber').value = tds[1].innerText.trim();
                document.getElementById('editSerialNumbers').value = tds[3].innerText.trim();
                // Store row index for later use
                document.getElementById('editRowIndex').value = row.rowIndex;
                new bootstrap.Modal(document.getElementById('editItemModal')).show();
            });
        });

        document.addEventListener('DOMContentLoaded', function() {
            // Function to handle product selection change
            function handleProductChange(select) {
                const row = select.closest('tr');
                const requireSerials = select.options[select.selectedIndex].dataset.requireSerials === '1';
                const serialColumn = row.querySelector('.serial-column');
                const serialContainer = row.querySelector('.serial-numbers-container');
                const serialButtons = row.querySelector('.serial-buttons');
                const serialInput = row.querySelector('.serial-numbers-input');
                
                if (requireSerials) {
                    serialColumn.style.display = '';
                    serialButtons.style.display = '';
                    serialInput.required = true;
                } else {
                    serialColumn.style.display = 'none';
                    serialButtons.style.display = 'none';
                    serialInput.required = false;
                    serialInput.value = '';
                }
            }

            // Add event listeners to existing product selects
            document.querySelectorAll('.product-select').forEach(select => {
                select.addEventListener('change', function() {
                    handleProductChange(this);
                });
            });



            // Add event listeners for remove buttons
            document.addEventListener('click', function(e) {
                if (e.target.classList.contains('remove-item')) {
                    const tbody = document.querySelector('#itemsTable tbody');
                    if (tbody.rows.length > 1) {
                        e.target.closest('tr').remove();
                    }
                }
            });

            // Initialize serial column visibility for existing rows
            document.querySelectorAll('.product-select').forEach(select => {
                handleProductChange(select);
            });
        });

        const nextBatchNumber = '<?php echo $next_batch_number_suggestion; ?>';
        const nextManualBatchNumber = '<?php echo $next_manual_batch_number; ?>';

        function getNextBatchNumberForNewRow() {
            const rows = document.querySelectorAll('#product-items .product-item');
            // Simple logic to increment suggestion for multiple new rows on client-side
            return String(parseInt(nextBatchNumber) + rows.length).padStart(5, '0');
        }

        function getNextManualBatchNumber() {
            return nextManualBatchNumber;
        }

        function addProductItem() {
            // ... existing code ...
            // Set the batch number for the new item
            const newBatchNumber = getNextBatchNumberForNewRow();
            newItem.querySelector('.batch-number').value = newBatchNumber;

            itemsContainer.appendChild(newItem);
        }

        // Set initial batch number
        const firstBatchInput = document.querySelector('.batch-number');
        if (firstBatchInput && !firstBatchInput.value) {
            firstBatchInput.value = getNextBatchNumberForNewRow();
        }

        // --- JavaScript for Supplier Grouping ---
        document.querySelectorAll('.supplier-header').forEach(header => {
            header.addEventListener('click', function() {
                const supplierId = this.dataset.supplierId;
                const receiptsDiv = document.querySelector(`.supplier-receipts[data-supplier-id="${supplierId}"]`);
                const toggleIcon = this.querySelector('.supplier-toggle');
                
                if (receiptsDiv.style.display === 'none') {
                    receiptsDiv.style.display = 'block';
                    toggleIcon.classList.remove('fa-chevron-right');
                    toggleIcon.classList.add('fa-chevron-down');
                } else {
                    receiptsDiv.style.display = 'none';
                    toggleIcon.classList.remove('fa-chevron-down');
                    toggleIcon.classList.add('fa-chevron-right');
                }
            });
        });

        // Filter functionality
        let currentFilter = 'all';

        function filterReceipts(filter) {
            currentFilter = filter;
            
            // Update button states
            document.querySelectorAll('[id^="filter-"]').forEach(btn => {
                btn.classList.remove('btn-primary', 'btn-outline-primary', 'btn-success', 'btn-outline-success', 'btn-warning', 'btn-outline-warning');
                if (btn.id === 'filter-' + filter) {
                    if (filter === 'all') {
                        btn.classList.add('btn-primary');
                    } else if (filter === 'added') {
                        btn.classList.add('btn-success');
                    } else if (filter === 'pending') {
                        btn.classList.add('btn-warning');
                    }
                } else {
                    if (btn.id === 'filter-all') {
                        btn.classList.add('btn-outline-primary');
                    } else if (btn.id === 'filter-added') {
                        btn.classList.add('btn-outline-success');
                    } else if (btn.id === 'filter-pending') {
                        btn.classList.add('btn-outline-warning');
                    }
                }
            });

            // Show/hide rows based on filter
            document.querySelectorAll('.receipt-row').forEach(row => {
                const status = row.dataset.status;
                
                if (filter === 'all' || status === filter) {
                    row.style.display = 'table-row';
                } else {
                    row.style.display = 'none';
                }
            });

            // Update supplier group visibility
            document.querySelectorAll('.supplier-group').forEach(group => {
                const visibleRows = group.querySelectorAll('.receipt-row[style*="table-row"], .receipt-row:not([style*="none"])');
                const allRows = group.querySelectorAll('.receipt-row');
                
                // Show supplier group if it has visible rows or if showing all
                if (filter === 'all' || visibleRows.length > 0) {
                    group.style.display = 'block';
                } else {
                    group.style.display = 'none';
                }
            });
        }

        // Initialize filter on page load
        document.addEventListener('DOMContentLoaded', function() {
            filterReceipts('all');
        });

        // --- End of Supplier Grouping JS ---
    </script>
    <?php include '../includes/footer.php'; ?>
</body>
</html> 